/**
 * \file: WaylandInputSourceImpl.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto
 *
 * \author: M. Adachi / ADIT/SW / madachi@jp.adit-jv.com
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <adit_logging.h>
#include "WaylandInputSourceImpl.h"

LOG_IMPORT_CONTEXT(aauto_input)

namespace adit { namespace aauto {

using namespace std;
using namespace uspi;

WaylandInputSourceImpl::WaylandInputSourceImpl(InputSource* inInputSource, void* inSessionContext)
: SharedDataReceiver(inSessionContext)
{
    inputSource = inInputSource;
    mCallbacks = nullptr;
    surface = nullptr;
    wayland = nullptr;

    running = false;
    display = nullptr;

    mMarginX = 0;
    mMarginY = 0;
}

WaylandInputSourceImpl::~WaylandInputSourceImpl()
{
    if (config.shareWlDisplay)
    {
        if (0 != SharedDataSender::instance().unsubscribeFromDataSharing(*this)) {
            LOG_ERROR((aauto_input, "~WaylandInputSourceImpl() could not unsubscribe from data sharing"));
        }
    }

    if (true == running) {
        if (!shutdown()) {
            LOG_WARN((aauto_input, "shutdown() failed"));
        }
    }
    release();
    LOGD_DEBUG((aauto_input, "WaylandInputSource de-initialize was done."));
}

bool WaylandInputSourceImpl::init()
{
    bool ret = true;

    // Check configuration error
    if(config.ResultConfig())
    {
        if (config.shareWlDisplay)
        {
            /* register InputSource endpoint to get notified by
             * VideoSink endpoint about wl_display */
            if (0 != SharedDataSender::instance().subscribeForDataSharing(*this)) {
                LOG_ERROR((aauto_input, "init() could not subscribe for data sharing"));
                ret = false;
            }
        }
        else
        {
            if (!touchInit(&config, inputSource))
            {
                LOG_ERROR((aauto_input, "could not initialize InputSource Adapter"));
                ret = false;
            }
            else
            {
                LOGD_DEBUG((aauto_input, "WaylandInputSource initialization is success."));
                running = true;
            }
        }
    }
    else
    {
        LOG_ERROR((aauto_input, "Check configuration error"));
        ret = false;
    }

    inputSource->registerCallbacks(this);

    return ret;
}

bool WaylandInputSourceImpl::shutdown()
{
    bool ret = true;

    if (true == running) {
        if (wayland) {
            wayland->StopInputThread();
        } else {
            LOG_ERROR((aauto_input, "WaylandInputSourceImpl::shutdown()  wayland is nullptr"));
            ret = false;
        }
        if (!inputReport.shutdown())
        {
            ret = false;
        } else {
            running = false;
        }
    } else {
        LOG_INFO((aauto_input, "WaylandInputSourceImpl::shutdown()  Not running. Was already stopped"));
    }

    LOGD_DEBUG((aauto_input, "WaylandInputSourceImpl shutdown done"));

    return ret;
}

void WaylandInputSourceImpl::setConfigItem(string inKey, string inValue)
{
    LOGD_DEBUG((aauto_input, "setConfigItem: inKey = %s inValue = %s", inKey.c_str(),inValue.c_str())); 
    config.set(inKey, inValue);
}

void WaylandInputSourceImpl::registerCallbacks(IAditInputSourceCallbacks* inCallbacks)
{
    mCallbacks = inCallbacks;
}

void WaylandInputSourceImpl::onSurfaceDataProvided(wl_display* inWlDisplay)
{
    if (running) {
        LOG_WARN((aauto_input, "onSurfaceDataProvided() WaylandInputSource already running"));
        return;
    }

    if (inWlDisplay != nullptr) {
        display = inWlDisplay;
    } else {
        LOGD_DEBUG((aauto_input, "onSurfaceDataProvided() Initialize touch with own surface"));
        display = nullptr;
    }

    if (!touchInit(&config, inputSource)) {
        LOG_ERROR((aauto_input, "onSurfaceDataProvided() could not initialize InputSource Adapter"));
        if (mCallbacks != nullptr)
        {
            mCallbacks->notifyErrorCallback(ATTACHED_TOUCH_INIT_ERROR);
        }
    } else {
        LOGD_DEBUG((aauto_input, "onSurfaceDataProvided() WaylandInputSource initialization is successful."));
        running = true;
    }
}

void WaylandInputSourceImpl::onSurfaceDataExpired()
{
    if (running) {
        /* shut down WaylandInputSourceImpl */
        if (!shutdown()) {
            LOG_WARN((aauto_input, "onSurfaceDataExpired() shutdown failed"));
        }
    } else {
        LOG_WARN((aauto_input, "onSurfaceDataExpired() WaylandInputSource already stopped"));
    }

    release();

    LOG_INFO((aauto_input, "onSurfaceDataExpired()  Expiration handled,"));
}

void WaylandInputSourceImpl::onResolutionDataProvided(int inWidth, int inHeight, int inMarginX, int inMarginY)
{
    /* This API is only received if shareWlDisplay was enabled.
       In such a case, we did not differentiate between touchWidth/touchHeight
       and isplayWidth/displayHeight. */
    config.displayWidth  = inWidth;
    config.displayHeight = inHeight;
    config.touchWidth    = inWidth;
    config.touchHeight   = inHeight;
    mMarginX = inMarginX;
    mMarginY = inMarginY;

    LOGD_DEBUG((aauto_input, "onResolutionDataProvided()  Set margin to %d:%d for resolution %dx%d",
                mMarginX, mMarginY, config.displayWidth, config.displayHeight));
}

void WaylandInputSourceImpl::onInputFeedback(const InputFeedback& inFeedback)
{
    if (mCallbacks != nullptr)
    {
        mCallbacks->onInputFeedback(inFeedback);
    } else {
        // registering for input callbacks is optional, at least for now
        LOG_INFO((aauto_input, "No callbacks registered"));
    }
}

bool WaylandInputSourceImpl::touchInit(InputSourceConfig* inConfig, InputSource* inInputSource)
{
    inputReport.initialize(inInputSource, inConfig->touchWidth, inConfig->touchHeight, mMarginX, mMarginY);

    // create wayland
    if (display) { // display only set if VideoSink endoint shares the wl_display
        LOGD_DEBUG((aauto_input, "Use wl_display from VideoSink"));
        /* use wl_display created by VideoSink endpoint */
        wayland = new WaylandContext(display, mCallbacks);
    } else {
        LOGD_DEBUG((aauto_input, "Use own InputSource wl_display"));
        wayland = new WaylandContext(mCallbacks);
    }

    aauto_return_value_on_null(aauto, wayland, false);
    if (inConfig->deviceType == AutoDetectType || inConfig->deviceType == SingleTouchDeviceType ||
        inConfig->deviceType == MultiTouchDeviceType)
    {
        // add touch device listener
        shared_ptr<WaylandContext::DeviceListener> touch(new TouchListener(inputReport, inConfig->displayWidth, inConfig->displayHeight, \
                                                                           inConfig->verbose, inConfig->deviceType));
        wayland->SetTouchListener(&TouchListener::Listener, move(touch));
    }

    if (inConfig->deviceType == AutoDetectType || inConfig->deviceType == PointerDeviceType)
    {
        // add pointer device listener
        shared_ptr<WaylandContext::DeviceListener> pointer(new PointerListener(inputReport, inConfig->displayWidth, inConfig->displayHeight, \
                                                                               inConfig->verbose));
        wayland->SetPointerListener(&PointerListener::Listener, move(pointer));
    }

    // init
    if (!wayland->Initialize())
    {
        LOG_ERROR((aauto_input, "Could not initialize wayland"));
        return false;
    }

    if (display == nullptr) { // display only set if VideoSink endoint shares the wl_display
        // create surface
        surface = new WaylandSurface();
        aauto_return_value_on_null(aauto_input, surface, false);
        if (!surface->Create(wayland, inConfig->displayWidth, inConfig->displayHeight, inConfig->layerId, inConfig->surfaceId))
        {
            LOG_ERROR((aauto_input, "Could not create surface with id = %d for input", inConfig->surfaceId));
            return false;
        }
    }

    // run input handler thread
    int triggerInterval = inConfig->triggerInterval;
    if (triggerInterval <= 0)
    {
        triggerInterval = 50;
        LOG_WARN((aauto_input, "invalid time interval for input, set default to %d usec", triggerInterval));
    }

    if (!wayland->StartInputThread(triggerInterval))
    {
        LOG_ERROR((aauto_input, "Could not start the Input thread"));
        return false;
    }

    return true;
}

void WaylandInputSourceImpl::release()
{
    /* surface created if it was not shared by VideoSink endpoint,
     * but we can call aauto_safe_delete because of internal nullptr check */
    aauto_safe_delete(surface);
    aauto_safe_delete(wayland);
}

} } // namespace adit { namespace aauto {
